	function [G, Z, R_ZEROS, P, Wsnew] = CA_POLES_S(Wc, Ws, Amax, Amin, N)	

	% 	Computes the denormalized poles, zeros and gain constant for  
	%	an analog lowpass Cauer filter.  
	%	The design margin is used to decrease the stopband edge.
	% 	This program which is based on the program of Orchard and Willson
	%	is faster and more accurate than the MATLAB version ellipap
	%
	% 	Toolbox for DIGITAL FILTERS USING MATLAB 
	
	% 	Author: 		Lars Wanhammar, 2002-07-23
    % 	Modified by: 	LW 2006-01-12, 2013-12-30, 2014-04-13 
	% 	Version: 		1
	% 	Known bugs:		
	% 	Report bugs to:	Wanhammar@gmail.com
	
	% 	References: 	H.J. Orchard and A.N. Willson, Jr: Elliptic Functions for Filter
	% 					Design, IEEE Trans. on Circuits and Systems, Part I, Vol. CAS-I 44,
	%					No. 4, pp.273-287, April 1997.	
	
	if floor(N) ~= -floor(-N)
		disp('N must be an interger')
		return
	end
 
	if (Wc >= Ws)
		error('Not a lowpass filter. Must have Wc < Ws')  
	elseif ( Amax <= 0 | Amax >= Amin)
		error('The passband attenuation must be 0 < Amax < Amin')
	elseif ( Amin <= 0)
		error('The stopband attenuation must be Amin > Amax > 0')
	end 
	
	if ~(abs(N-round(N)) < eps)
		N = floor(CA_ORDER_S(Wc, Ws, Amax, Amin)) + 1;
		return
	end
	if N == 1	% Special case 
		Z = [];
		Apn = Amax*log(10)/20;
		epsilon = sqrt(2*exp(Apn)*sinh(Apn));
		P = -Wc*epsilon^(-1/N);
		G = -P;
		return
	end
	if (Wc >= Ws)
		error('Not a lowpass filter. Must have Wc < Ws')  
	elseif ( Amax <= 0 | Amax >= Amin)
		error('The passband attenuation must be 0 < Amax < Amin')
	elseif ( Amin <= 0)
		error('The stopband attenuation must be Amin > Amax > 0')
	end 
	N0 = rem(N, 2);
	N3 = (N - N0)/2;
	Apn = Amax*log(10)/20;
	Asn = Amin*log(10)/20;
	epsilon2 = 2*exp(Apn)*sinh(Apn);
	e(1) = sqrt(epsilon2);
	g(1) = e(1)/sqrt(exp(2*Asn) - 1);
	v = g(1);
	m2 = 1;
	while v > 1.e-150
		v = (v/(1 + sqrt(1 - v*v)))^2;
		m2 = m2 + 1;
		g(m2) = v;
	end
	for index = 1:10
		m1 = m2 + index;
		ek(m1) = 4*(g(m2)/4)^((2^index)/N);
		if (ek(m1) < 100*eps), break, end
	end
	for en = m1:-1:2
		ek(en-1) = 2*sqrt(ek(en))/(1 + ek(en));
	end
	% Compute the poles and zeros
	for en = 2:m2
		a = (1 + g(en))*e(en-1)/2;
		e(en) = a + sqrt(a*a + g(en));
	end
	u2 = log((1 + sqrt(1 + e(m2)*e(m2)))/e(m2))/N;
	Z = [];
	P = [];	
	for index = 1:N3
		u1 = (2*index-1)*pi/(2*N);
		c = -i/cos((u2*i - u1));
		d = 1/cos(u1);
		for en = m1:-1:2
			c = (c - ek(en)/c)/(1 + ek(en));
			d = (d + ek(en)/d)/(1 + ek(en));
		end
		af(index) = 1/c;
		df(index) = d/ek(1);
		P = [conj(af(index)); af(index); P];
		Z = [-df(index)*i; df(index)*i; Z];
	end
	if N0 == 1
		a = 1/sinh(u2);
		for en = m1:-1:2
			a = (a - ek(en)/a)/(1 + ek(en));
		end
		P = [P; -1/a];
	end

	% Denormalize
	Z = Wc*Z;
	P = Wc*P;
	G = real(prod(-P)/prod(-Z));	% Gain constant
	if (~rem(N, 2)) 				% N is even, hence correct the gain
		G = G/sqrt(1 + epsilon2);
	end

	% Determine the true stopband edge for the type A filter of order N 
	%	 can be done nicer!
	delta = (Ws-Wc)/20; Wsnew = Wc;
	for n = 1:100
		W = Wsnew + delta;
		A = PZ_2_ATT_S(G, Z, P, W);
		if A < Amin
			Wsnew = W;
		else
			delta = delta/5;
		end
	end
	% The reflection zeros are
	R_ZEROS = cat(1, (Wsnew./Z)*Wc );
	if rem(N,2) ~= 0
		R_ZEROS = cat(1,R_ZEROS,[0]);
	end
